--
-- KuhnGA4321GM
-- Specialization for KuhnGA4321GM
--
-- @author  Manuel Leithner
-- @date  26/07/09
--

KuhnGA4321GM = {};

function KuhnGA4321GM.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Attachable, specializations);
end;

function KuhnGA4321GM:load(xmlFile)
	
	self.setDamperDirection = SpecializationUtil.callSpecializationsFunction("setDamperDirection");

	self.groundReferenceThreshold = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.groundReferenceNode#threshold"), 0.2);
    self.groundReferenceNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.groundReferenceNode#index"));
    if self.groundReferenceNode == nil then
        self.groundReferenceNode = self.components[1].node;
    end;

    self.animation = {};
    self.animation.animCharSet = 0;
    self.animationEnabled = false;

    local rootNode = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.animation#rootNode"));

    if rootNode ~= nil then
        self.animation.animCharSet = getAnimCharacterSet(rootNode);
        if self.animation.animCharSet ~= 0 then
            self.animation.clip = getAnimClipIndex(self.animation.animCharSet, getXMLString(xmlFile, "vehicle.animation#animationClip"));
            if self.animation.clip >= 0 then
                assignAnimTrackClip(self.animation.animCharSet, 0, self.animation.clip);
                self.animation.speedScale = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.animation#speedScale"), 1);
                setAnimTrackSpeedScale(self.animation.animCharSet, self.animation.clip, self.animation.speedScale);
                setAnimTrackLoopState(self.animation.animCharSet, 0, true);
            end;
        end;
    end;

    local numWindrowerDropAreas = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.windrowerDropAreas#count"), 0);
    if numWindrowerDropAreas ~= 1 or numWindrowerDropAreas ~= table.getn(self.cuttingAreas) then
        print("Warning: number of cutting areas and drop areas should be equal");
    end;
    self.windrowerDropAreas = {}
    for i=1, numWindrowerDropAreas do
        self.windrowerDropAreas[i] = {};
        local areanamei = string.format("vehicle.windrowerDropAreas.windrowerDropArea%d", i);
        self.windrowerDropAreas[i].start = Utils.indexToObject(self.components, getXMLString(xmlFile, areanamei .. "#startIndex"));
        self.windrowerDropAreas[i].width = Utils.indexToObject(self.components, getXMLString(xmlFile, areanamei .. "#widthIndex"));
        self.windrowerDropAreas[i].height = Utils.indexToObject(self.components, getXMLString(xmlFile, areanamei .. "#heightIndex"));
    end;

    local numCuttingAreas = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.cuttingAreas#count"), 0);
    for i=1, numCuttingAreas do
        local areanamei = string.format("vehicle.cuttingAreas.cuttingArea%d", i);
        self.cuttingAreas[i].foldMinLimit = Utils.getNoNil(getXMLFloat(xmlFile, areanamei .. "#foldMinLimit"), 0);
        self.cuttingAreas[i].foldMaxLimit = Utils.getNoNil(getXMLFloat(xmlFile, areanamei .. "#foldMaxLimit"), 1);
    end;

    local windrowerSound = getXMLString(xmlFile, "vehicle.windrowerSound#file");
    if windrowerSound ~= nil and windrowerSound ~= "" then
        windrowerSound = Utils.getFilename(windrowerSound, self.baseDirectory);
        self.windrowerSound = createSample("windrowerSound");
        self.windrowerSoundEnabled = false;
        loadSample(self.windrowerSound, windrowerSound, false);
        self.windrowerSoundPitchOffset = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.windrowerSound#pitchOffset"), 1);
        self.windrowerSoundVolume = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.windrowerSound#volume"), 1.0);
    end;

	self.particleSystems = {};
    local i=0;
    while true do
        local baseName = string.format("vehicle.particleSystems.particleSystem(%d)", i);

        local particleSystem = {};
        particleSystem.ps = {};
        local ps = Utils.loadParticleSystem(xmlFile, particleSystem.ps, baseName, self.components[2].node, false, nil, self.baseDirectory)
        if ps == nil then
            break;
        end;
        particleSystem.disableTime = 0;
        table.insert(self.particleSystems, particleSystem);
        i = i+1;
    end;
	
	
	local dampersCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.dampers#count"), 0);	
	self.dampers = {};	
	for i=1, dampersCount do
		local damperName = string.format("vehicle.dampers.damper%d", i);		
		self.dampers[i] = {};		
		self.dampers[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, damperName .. "#index"));
		self.dampers[i].part = Utils.indexToObject(self.components, getXMLString(xmlFile, damperName .. "#part"));
		self.dampers[i].damperFixpoint = Utils.indexToObject(self.components, getXMLString(xmlFile, damperName .. "#damperFixpoint"));
		self.dampers[i].fixPoint = Utils.indexToObject(self.components, getXMLString(xmlFile, damperName .. "#fixpoint"));
		local ax, ay, az = getWorldTranslation(self.dampers[i].part);
		local bx, by, bz = getWorldTranslation(self.dampers[i].damperFixpoint);		
		self.dampers[i].partDistance = Utils.vector3Length(ax-bx, ay-by, az-bz);	
	end;	
	
	local safetyBarCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.safetyBars#count"), 0);	
	self.safetyBars = {};	
	for i=1, safetyBarCount do
		local safetyBarName = string.format("vehicle.safetyBars.safetyBar%d", i);		
		self.safetyBars[i] = {};		
		self.safetyBars[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, safetyBarName .. "#index"));
		local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, safetyBarName .. "#minRot"));
		self.safetyBars[i].minRot = {};
		self.safetyBars[i].minRot[1] = Utils.degToRad(x);
		self.safetyBars[i].minRot[2] = Utils.degToRad(y);
		self.safetyBars[i].minRot[3] = Utils.degToRad(z);
		x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, safetyBarName .. "#maxRot"));
		self.safetyBars[i].maxRot = {};
		self.safetyBars[i].maxRot[1] = Utils.degToRad(x);
		self.safetyBars[i].maxRot[2] = Utils.degToRad(y);
		self.safetyBars[i].maxRot[3] = Utils.degToRad(z);		
		self.safetyBars[i].moveTime = Utils.getNoNil(getXMLFloat(xmlFile, safetyBarName .. "#moveTime"), 3) * 1000;
	end;

	self.swathBoard = {};
	self.swathBoard.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.swathBoard#index"));
	local x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#minRot"));
	self.swathBoard.minRot = {};
	self.swathBoard.minRot[1] = Utils.degToRad(x);
	self.swathBoard.minRot[2] = Utils.degToRad(y);
	self.swathBoard.minRot[3] = Utils.degToRad(z);
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#maxRot"));
	self.swathBoard.maxRot = {};
	self.swathBoard.maxRot[1] = Utils.degToRad(x);
	self.swathBoard.maxRot[2] = Utils.degToRad(y);
	self.swathBoard.maxRot[3] = Utils.degToRad(z);
	self.swathBoard.moveTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.swathBoard#moveTime"), 3) * 1000;
	self.swathBoard.arm1 = {};
	self.swathBoard.arm1.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.swathBoard#indexArm1"));
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#minRot1"));
	self.swathBoard.arm1.minRot = {};
	self.swathBoard.arm1.minRot[1] = Utils.degToRad(x);
	self.swathBoard.arm1.minRot[2] = Utils.degToRad(y);
	self.swathBoard.arm1.minRot[3] = Utils.degToRad(z);
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#maxRot1"));
	self.swathBoard.arm1.maxRot = {};
	self.swathBoard.arm1.maxRot[1] = Utils.degToRad(x);
	self.swathBoard.arm1.maxRot[2] = Utils.degToRad(y);
	self.swathBoard.arm1.maxRot[3] = Utils.degToRad(z);
	self.swathBoard.arm2 = {};
	self.swathBoard.arm2.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.swathBoard#indexArm2"));
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#minRot2"));
	self.swathBoard.arm2.minRot = {};
	self.swathBoard.arm2.minRot[1] = Utils.degToRad(x);
	self.swathBoard.arm2.minRot[2] = Utils.degToRad(y);
	self.swathBoard.arm2.minRot[3] = Utils.degToRad(z);
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#maxRot2"));
	self.swathBoard.arm2.maxRot = {};
	self.swathBoard.arm2.maxRot[1] = Utils.degToRad(x);
	self.swathBoard.arm2.maxRot[2] = Utils.degToRad(y);
	self.swathBoard.arm2.maxRot[3] = Utils.degToRad(z);
	self.swathBoard.board = {};
	self.swathBoard.board.node = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.swathBoard#swathBoard"));
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#minRot3"));
	self.swathBoard.board.minRot = {};
	self.swathBoard.board.minRot[1] = Utils.degToRad(x);
	self.swathBoard.board.minRot[2] = Utils.degToRad(y);
	self.swathBoard.board.minRot[3] = Utils.degToRad(z);
	x,y,z = Utils.getVectorFromString(getXMLString(xmlFile, "vehicle.swathBoard#maxRot3"));
	self.swathBoard.board.maxRot = {};
	self.swathBoard.board.maxRot[1] = Utils.degToRad(x);
	self.swathBoard.board.maxRot[2] = Utils.degToRad(y);
	self.swathBoard.board.maxRot[3] = Utils.degToRad(z);
	
	local tineCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.tines#count"), 0);	
	self.tines = {};
	self.transportTines = {};
	self.transportTime = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.tines#transportTime"), 3) * 1000;
	for i=1, tineCount do
		local tineName = string.format("vehicle.tines.tine%d", i);		
		self.tines[i] = {};		
		self.tines[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, tineName .. "#index"));
		self.transportTines[i] = {};
		self.transportTines[i].node = Utils.indexToObject(self.components, getXMLString(xmlFile, tineName .. "#transport"));
	end;
	
	self.invisibleTineCount = Utils.getNoNil(getXMLInt(xmlFile, "vehicle.tines#visibleCount"), 0);
	self.timePerTine = self.transportTime / table.getn(self.transportTines);
	self.timeToNextTine = self.timePerTine;
	self.doChangesForTransport = false;
	
	self.streetMarker = {};
	self.streetMarker.left = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.streetMarker#indexLeft"));
	self.streetMarker.right = Utils.indexToObject(self.components, getXMLString(xmlFile, "vehicle.streetMarker#indexRight"));
	self.streetMarker.maxDistance = Utils.getNoNil(getXMLFloat(xmlFile, "vehicle.streetMarker#maxDistance"), 1.6);
	
	self.rotLimit = {10, 25, 0};

	self.isTransport = true;
	self.attacherVehicleJoint = nil;
	self.mowerIsAttached = false;	
    self.isTurnedOn = false;
    self.wasToFast = false;
	self.isExpanded = false;
	
end;

function KuhnGA4321GM:delete()
    if self.windrowerSound ~= nil then
        delete(self.windrowerSound);
    end;
end;

function KuhnGA4321GM:mouseEvent(posX, posY, isDown, isUp, button)
end;

function KuhnGA4321GM:keyEvent(unicode, sym, modifier, isDown)
end;

function KuhnGA4321GM:update(dt)
	
	if self.mowerIsAttached then
		for i=1, table.getn(self.attacherVehicle.attachedImplements) do
			if self.attacherVehicle.attachedImplements[i].object == self then		
				local index = self.attacherVehicle.attachedImplements[i].jointDescIndex;
				self.attacherVehicleJoint = self.attacherVehicle.attacherJoints[index];	
				break;				
			end;
		end;
		self.mowerIsAttached = false;
	end;
	
	if self:getIsActive() then
		
		self.wasToFast = false;
		self.isExpanded = false;
		
		local bar = self.safetyBars[1];
		local x,y,z = getRotation(bar.node);
		if math.abs(x - bar.maxRot[1]) < 0.01 and math.abs(y - bar.maxRot[2]) < 0.01 and math.abs(z - bar.maxRot[3]) < 0.01 then
			self.isExpanded = true;
		end;		
		
		for k,v in pairs(self.particleSystems) do
            if self.time > v.disableTime then
                Utils.setEmittingState(v.ps, false);
            end;
        end;
		
		if self.doChangesForTransport then
			
			self.timeToNextTine = self.timeToNextTine - dt;
			
			if self.timeToNextTine < 0 then	
				
				self.timeToNextTine = self.timePerTine;			
				if self.isTransport then
					local xLeft, yLeft, zLeft = getWorldTranslation(self.streetMarker.left);
					local xRight, yRight, zRight = getWorldTranslation(self.streetMarker.right);
					local maxDistance = self.streetMarker.maxDistance;
					local isCheckComplete = true;
					
					for i=1, table.getn(self.tines) do						
						-- calculate distance to left and right streetmarks  to disable tines 
						local x,y,z = getWorldTranslation(self.tines[i].node);
						local distanceLeft = Utils.vector3Length(x-xLeft, y-yLeft, z-zLeft);
						local distanceRight = Utils.vector3Length(x-xRight, y-yRight, z-zRight);
						local isVisible = getVisibility(self.tines[i].node);
						if (distanceLeft < maxDistance or distanceRight < maxDistance) and isVisible then
							setVisibility(self.tines[i].node, false);
							self.invisibleTineCount = self.invisibleTineCount + 1;
							setVisibility(self.transportTines[self.invisibleTineCount].node, true);
							isCheckComplete = false;
							break;
						end;						
					end;
					
					if isCheckComplete then
						self.doChangesForTransport = false;
					end;
				else					
					if self.invisibleTineCount > 0 then
					
						local isCheckComplete = true;
						for i=1, table.getn(self.tines) do							
							local isVisible = getVisibility(self.tines[i].node);							
							if not isVisible then								
								setVisibility(self.tines[i].node, true);	
								setVisibility(self.transportTines[self.invisibleTineCount].node, false);
								self.invisibleTineCount = self.invisibleTineCount - 1;
								isCheckComplete = false;
								break;
							end;						
						end;	
						
						if isCheckComplete then
							self.doChangesForTransport = false;
						end;
					end;
				end;
			end;
		end;	
		
	    if self:getIsActiveForInput() then
	        if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA) and not self.isTransport and self.isExpanded then
	            self.isTurnedOn = not self.isTurnedOn;
	        end;
			
			if InputBinding.hasEvent(InputBinding.IMPLEMENT_EXTRA2) then
				self.isTransport = not self.isTransport;
				self.doChangesForTransport = true;	
				
				if self.isTransport then
					self.isTurnedOn = false;
				end;
			end;
	    end;	

		if self.isTurnedOn then
		
			local toFast = false;
			local x,y,z = getWorldTranslation(self.groundReferenceNode);
			local terrainHeight = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, x, 0, z);	

            if terrainHeight+self.groundReferenceThreshold >= y then							
				
				toFast = self:doCheckSpeedLimit() and self.lastSpeed*3600 > 31;				
	            if not toFast then
                    local numDropAreas = table.getn(self.windrowerDropAreas);
                    local numAreas = table.getn(self.cuttingAreas);
                    local sum = 0;
                    local fruitType = FruitUtil.FRUITTYPE_GRASS;					
                    local fruitTypeFix = false;
					for i=1, 3 do 
					
						if i==2 then
							fruitType = FruitUtil.FRUITTYPE_WHEAT;
						elseif i==3 then
							fruitType = FruitUtil.FRUITTYPE_BARLEY;
						end;
						
	                    for j=1, numAreas do
	                        local cuttingArea = self.cuttingAreas[j];
	                        if self.isExpanded then
	                            local x,y,z = getWorldTranslation(cuttingArea.start);
	                            local x1,y1,z1 = getWorldTranslation(cuttingArea.width);
	                            local x2,y2,z2 = getWorldTranslation(cuttingArea.height);

	                            local ratio = g_currentMission.windrowCutLongRatio;

	                            if not fruitTypeFix and i==1 then
	                                fruitType = FruitUtil.FRUITTYPE_GRASS;
	                            end;

	                            local area = Utils.updateFruitCutLongArea(fruitType, x, z, x1, z1, x2, z2, 0);
	                            area = area + Utils.updateFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2, 0)*ratio;
								
	                            if area == 0 and not fruitTypeFix and i==1 then
	                                fruitType = FruitUtil.FRUITTYPE_DRYGRASS;
	                                area = Utils.updateFruitCutLongArea(fruitType, x, z, x1, z1, x2, z2, 0);
	                                area = area + Utils.updateFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2, 0)*ratio;
	                            end;

	                            if area > 0 then
	                                fruitTypeFix = true;
									Utils.switchFruitTypeArea(FruitUtil.FRUITTYPE_GRASS, FruitUtil.FRUITTYPE_DRYGRASS, x, z, x1, z1, x2, z2, 1);
	                            end;

	                            if numDropAreas >= numAreas then
	                                if area > 0 then
	                                    local dropArea = self.windrowerDropAreas[j];
	                                    local x,y,z = getWorldTranslation(dropArea.start);
	                                    local x1,y1,z1 = getWorldTranslation(dropArea.width);
	                                    local x2,y2,z2 = getWorldTranslation(dropArea.height);
	                                    local old, total = Utils.getFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2);
	                                    area = area + old;
	                                    local value = area / total;
	                                    if value < 1 and value > 0.08 then
	                                        value = 1;
	                                    else
	                                        value = math.floor(value + 0.6); -- round, biased to the bigger value
	                                    end;
	                                    if value >= 1 then
	                                        value = math.min(value, g_currentMission.maxWindrowValue);
	                                        Utils.updateFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2, value, true, false);
	                                    end;									
													
										if g_currentMission.environment.lastRainScale <= 0.1 and g_currentMission.environment.timeSinceLastRain > 30 then
											ps = self.particleSystems[1];
											if ps ~= nil then
												ps.disableTime = self.time + 20;
												Utils.setEmittingState(ps.ps, true);
											end;	
										end;
										
	                                end;
	                            else
	                                sum = sum + area;
	                            end;
	                        end;
	                    end;
	                    if sum > 0 and numDropAreas > 0 then
	                        local dropArea = self.windrowerDropAreas[1];
	                        local x,y,z = getWorldTranslation(dropArea.start);
	                        local x1,y1,z1 = getWorldTranslation(dropArea.width);
	                        local x2,y2,z2 = getWorldTranslation(dropArea.height);
	                        local old, total = Utils.getFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2);
	                        sum = sum + old;
	                        local value = math.floor(sum / total + 0.7); -- round, biased to the bigger value
	                        if value >= 1 then
	                            value = math.min(value, g_currentMission.maxWindrowValue);
	                            Utils.updateFruitWindrowArea(fruitType, x, z, x1, z1, x2, z2, value, true, false);
	                        end;							
	                    end;
					end;
                end;
            end;

            if not self.animationEnabled then
                enableAnimTrack(self.animation.animCharSet, 0);
                self.animationEnabled = true;
            end;

            if not self.windrowerSoundEnabled and self:getIsActiveForSound() then
                playSample(self.windrowerSound, 0, self.windrowerSoundVolume, 0);
                setSamplePitch(self.windrowerSound, self.windrowerSoundPitchOffset);
                self.windrowerSoundEnabled = true;
            end;

            self.wasToFast = toFast;
			
        else
            if self.animationEnabled then
                disableAnimTrack(self.animation.animCharSet, 0);
                self.animationEnabled = false;
            end;
        end;
		
		
		if self.attacherVehicleJoint ~= nil then
		
			local x, y, z = getRotation(self.attacherVehicleJoint.rotationNode);
			local maxRot = self.attacherVehicleJoint.maxRot;	
			local minRot = self.attacherVehicleJoint.minRot;
			local scale = self.attacherVehicleJoint.topArm.zScale;
			local current = 0;
			
			local xSection = 0;
			local ySection = 0;
			local zSection = 0;
			local angleRatio = 0;
			
			
			
			if maxRot[1] ~= 0 then		
				local minX = math.floor(math.deg(minRot[1]));
				local maxX = math.floor(math.deg(maxRot[1]));
				
				local angleSum = math.abs(minX) + math.abs(maxX);
				
				x = math.floor(math.deg(x));

				if scale == -1 then
					if x <= 0 then
						current = math.abs(minX) + math.abs(x);
					else
						current = math.abs(minX) - math.abs(x);
					end;
				else
					if x >= 0 then
						current = math.abs(minX) + math.abs(x);
					else
						current = math.abs(minX) - math.abs(x);
					end;			
				end;
				
				angleRatio = current / angleSum;
			end;		
			
			xSection = math.abs(angleRatio * self.rotLimit[1]);
			ySection = math.abs(angleRatio * self.rotLimit[2]);
			zSection = math.abs(angleRatio * self.rotLimit[3]);
			
			setJointRotationLimit(self.componentJoints[1].jointIndex, 0, true, Utils.degToRad(-xSection), Utils.degToRad(xSection));
			setJointRotationLimit(self.componentJoints[1].jointIndex, 1, true, Utils.degToRad(-ySection), Utils.degToRad(ySection));
			setJointRotationLimit(self.componentJoints[1].jointIndex, 2, true, Utils.degToRad(-zSection), Utils.degToRad(zSection));		
		end;

		self:setDamperDirection();	
		
		for i=1, table.getn(self.safetyBars) do			
			local safetyBar = self.safetyBars[i];
			local x,y,z = getRotation(safetyBar.node);
			local newRot = Utils.getMovedLimitedValues({x,y,z}, safetyBar.maxRot, safetyBar.minRot, 3, safetyBar.moveTime, dt, self.isTransport);
			setRotation(safetyBar.node, unpack(newRot));	
		end;	
	
		-- SwathBoard animation
		local x,y,z = getRotation(self.swathBoard.node);
		local newRot = Utils.getMovedLimitedValues({x,y,z}, self.swathBoard.maxRot, self.swathBoard.minRot, 3, self.swathBoard.moveTime, dt, self.isTransport);
		setRotation(self.swathBoard.node, unpack(newRot));
		x,y,z = getRotation(self.swathBoard.arm1.node);
		newRot = Utils.getMovedLimitedValues({x,y,z}, self.swathBoard.arm1.maxRot, self.swathBoard.arm1.minRot, 3, self.swathBoard.moveTime, dt, self.isTransport);
		setRotation(self.swathBoard.arm1.node, unpack(newRot));
		x,y,z = getRotation(self.swathBoard.arm2.node);
		newRot = Utils.getMovedLimitedValues({x,y,z}, self.swathBoard.arm2.maxRot, self.swathBoard.arm2.minRot, 3, self.swathBoard.moveTime, dt, self.isTransport);
		setRotation(self.swathBoard.arm2.node, unpack(newRot));
		x,y,z = getRotation(self.swathBoard.board.node);
		newRot = Utils.getMovedLimitedValues({x,y,z}, self.swathBoard.board.maxRot, self.swathBoard.board.minRot, 3, self.swathBoard.moveTime, dt, self.isTransport);
		setRotation(self.swathBoard.board.node, unpack(newRot));
	end;
	
	if not self.isTurnedOn and self.windrowerSoundEnabled then
        stopSample(self.windrowerSound);
        self.windrowerSoundEnabled = false;
    end;
	
end;

function KuhnGA4321GM:draw()
    if self.isTransport then
        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("transport_off"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA2);
    else
        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("transport_on"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA2);
		if self.isExpanded then
			if self.isTurnedOn then
		        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_off_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		    else
		        g_currentMission:addHelpButtonText(string.format(g_i18n:getText("turn_on_OBJECT"), self.typeDesc), InputBinding.IMPLEMENT_EXTRA);
		    end;
		end;
    end;

    if self.wasToFast then
        g_currentMission:addWarning(g_i18n:getText("Dont_drive_to_fast") .. "\n" .. string.format(g_i18n:getText("Cruise_control_levelN"), "2", InputBinding.getKeyNamesOfDigitalAction(InputBinding.SPEED_LEVEL2)), 0.07+0.022, 0.019+0.029);
    end;
end;

function KuhnGA4321GM:setDamperDirection()
	for i=1, table.getn(self.dampers) do
		local ax, ay, az = getWorldTranslation(self.dampers[i].node);
		local bx, by, bz = getWorldTranslation(self.dampers[i].fixPoint);
		local x, y, z = worldDirectionToLocal(getParent(self.dampers[i].node), bx-ax, by-ay, bz-az);
		
		setDirection(self.dampers[i].node, x, y, z, 0, 1, 0);
		if self.dampers[i].part ~= nil then
			local distance = Utils.vector3Length(ax-bx, ay-by, az-bz);
			setTranslation(self.dampers[i].part, 0, 0, distance-self.dampers[i].partDistance);
		end;	
	end;
end;



function KuhnGA4321GM:onAttach(attacherVehicle)
	self.mowerIsAttached = true;
	
	self.isBraking = true;
	for k,wheel in pairs(self.wheels) do
		setWheelShapeProps(wheel.node, wheel.wheelShape, 0, 15, wheel.steeringAngle);
	end;
end;

function KuhnGA4321GM:onDetach()
	setJointRotationLimit(self.componentJoints[1].jointIndex, 0, true, 0, 0);
	setJointRotationLimit(self.componentJoints[1].jointIndex, 1, true, 0, 0);
	setJointRotationLimit(self.componentJoints[1].jointIndex, 2, true, 0, 0);
	self:setDamperDirection();
	self.attacherVehicleJoint = nil;
	
	if self.deactivateOnDetach then
        KuhnGA4321GM.onDeactivate(self);
    end;
end;

function KuhnGA4321GM:onDeactivate()
	if self.animationEnabled then
        disableAnimTrack(self.animation.animCharSet, 0);
        self.animationEnabled = false;
    end;
	for k,v in pairs(self.particleSystems) do
        Utils.setEmittingState(v.ps, false);
    end;
    self.isTurnedOn = false;
end;


function KuhnGA4321GM:onLeave()
    if self.deactivateOnLeave then
        KuhnGA4321GM.onDeactivate(self);
    end;
end;


